import React, { useState, useRef } from 'react';
import { TrendingUp, TrendingDown, Activity, AlertCircle, DollarSign, BarChart3, Loader2, Search, Calendar, Shield } from 'lucide-react';
import StrategyTabs from './components/StrategyTabs';
import MetricsDisplay from './components/MetricsDisplay';
import InfoCards from './components/InfoCards';
import { analyzeStock } from './services/api';

function App() {
  const [ticker, setTicker] = useState('');
  const [analysis, setAnalysis] = useState(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState('');
  const inputRef = useRef(null);

  const handleAnalyze = async () => {
    if (!ticker.trim()) return;
    
    setLoading(true);
    setError('');
    
    try {
      const result = await analyzeStock(ticker.toUpperCase());
      setAnalysis(result);
    } catch (err) {
      setError('Failed to analyze stock. Please check the ticker symbol and try again.');
      console.error(err);
    } finally {
      setLoading(false);
    }
  };

  const handleKeyPress = (e) => {
    if (e.key === 'Enter') {
      handleAnalyze();
    }
  };

  return (
    <div className="min-h-screen gradient-bg text-white p-4 md:p-6">
      <header className="max-w-7xl mx-auto mb-8">
        <div className="flex flex-col md:flex-row justify-between items-center mb-6">
          <div>
            <h1 className="text-3xl md:text-4xl font-bold bg-clip-text text-transparent bg-gradient-to-r from-cyan-400 to-blue-500">
              Options Strategy Analyzer
            </h1>
            <p className="text-gray-300 mt-2">Professional-grade stock options analysis in real-time</p>
          </div>
          <div className="mt-4 md:mt-0">
            <div className="relative">
              <div className="flex items-center">
                <input
                  ref={inputRef}
                  type="text"
                  value={ticker}
                  onChange={(e) => setTicker(e.target.value.toUpperCase())}
                  onKeyPress={handleKeyPress}
                  placeholder="Enter stock ticker (e.g., AAPL, TSLA)"
                  className="w-full md:w-80 px-4 py-3 rounded-lg glass focus:border-cyan-400 focus:ring-2 focus:ring-cyan-400/30 focus:outline-none transition-all"
                />
                <button
                  onClick={handleAnalyze}
                  disabled={loading || !ticker.trim()}
                  className="ml-2 px-6 py-3 bg-gradient-to-r from-cyan-500 to-blue-600 hover:from-cyan-600 hover:to-blue-700 rounded-lg font-semibold transition-all disabled:opacity-50 disabled:cursor-not-allowed flex items-center"
                >
                  {loading ? (
                    <Loader2 className="w-5 h-5 animate-spin" />
                  ) : (
                    <>
                      <Search className="w-5 h-5 mr-2" />
                      Analyze
                    </>
                  )}
                </button>
              </div>
            </div>
            <p className="text-sm text-gray-400 mt-2">Press Enter to analyze</p>
          </div>
        </div>
      </header>

      <main className="max-w-7xl mx-auto">
        {error && (
          <div className="mb-6 p-4 bg-red-900/30 border border-red-700 rounded-lg flex items-center">
            <AlertCircle className="w-5 h-5 mr-2 text-red-400" />
            <span>{error}</span>
          </div>
        )}

        {loading && (
          <div className="flex flex-col items-center justify-center py-20">
            <Loader2 className="w-12 h-12 animate-spin text-cyan-400 mb-4" />
            <p className="text-lg">Analyzing {ticker}...</p>
            <p className="text-gray-400">Fetching real-time data and generating strategies</p>
          </div>
        )}

        {analysis && !loading && (
          <>
            <div className="mb-8">
              <div className="flex items-center justify-between">
                <div>
                  <h2 className="text-2xl font-bold">
                    {analysis.stockInfo.symbol} - {analysis.stockInfo.companyName}
                  </h2>
                  <div className="flex items-center mt-2">
                    <span className="text-3xl font-bold">${analysis.fundamentalMetrics.currentPrice}</span>
                    <span className={`ml-4 px-3 py-1 rounded-full text-sm font-semibold ${analysis.technicalAnalysis.trend === 'Bullish' ? 'bg-green-900/40 text-green-300' : analysis.technicalAnalysis.trend === 'Bearish' ? 'bg-red-900/40 text-red-300' : 'bg-gray-700 text-gray-300'}`}>
                      {analysis.technicalAnalysis.trend} Trend
                    </span>
                  </div>
                </div>
                <div className="text-right">
                  <p className="text-gray-400">Last Updated</p>
                  <p>{new Date().toLocaleTimeString()}</p>
                </div>
              </div>
            </div>

            <MetricsDisplay analysis={analysis} />
            <StrategyTabs strategies={analysis.strategies} />
            <InfoCards analysis={analysis} />

            <div className="mt-12 p-4 border border-white/10 rounded-lg bg-black/20">
              <div className="flex items-start">
                <Shield className="w-5 h-5 mr-2 text-amber-400 mt-0.5" />
                <div>
                  <p className="text-amber-400 font-semibold mb-1">Disclaimer</p>
                  <p className="text-sm text-gray-400">
                    This application provides educational information only and is not financial advice. Options trading involves significant risk and may not be suitable for all investors. Past performance is not indicative of future results. Always conduct your own research and consult with a qualified financial advisor before making investment decisions.
                  </p>
                </div>
              </div>
            </div>
          </>
        )}

        {!analysis && !loading && (
          <div className="text-center py-20">
            <div className="inline-block p-6 rounded-full bg-white/5 mb-6">
              <BarChart3 className="w-16 h-16 text-cyan-400" />
            </div>
            <h3 className="text-2xl font-semibold mb-4">Start Analyzing Options Strategies</h3>
            <p className="text-gray-400 max-w-lg mx-auto">
              Enter a stock ticker symbol above to get real-time analysis, metrics, and personalized options strategy recommendations.
            </p>
            <div className="mt-8 grid grid-cols-1 md:grid-cols-3 gap-4 max-w-2xl mx-auto">
              <div className="p-4 rounded-lg glass">
                <TrendingUp className="w-8 h-8 text-green-400 mb-2" />
                <p className="font-semibold">Wheel Strategy</p>
                <p className="text-sm text-gray-400">Sell puts & covered calls</p>
              </div>
              <div className="p-4 rounded-lg glass">
                <Activity className="w-8 h-8 text-blue-400 mb-2" />
                <p className="font-semibold">Directional Plays</p>
                <p className="text-sm text-gray-400">Buy calls or puts</p>
              </div>
              <div className="p-4 rounded-lg glass">
                <DollarSign className="w-8 h-8 text-purple-400 mb-2" />
                <p className="font-semibold">Credit Spreads</p>
                <p className="text-sm text-gray-400">Defined risk strategies</p>
              </div>
            </div>
          </div>
        )}
      </main>
    </div>
  );
}

export default App;